/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#if CROSS_PLATFORM
using Clock.Interfaces;
#endif
using Clock.Tizen;
using System;
using Xamarin.Forms;

namespace Clock.Alarm
{
    /// <summary>
    /// This class defines alarm list page which shows all alarms in ListView
    /// This inherits from floating button enabled content page
    /// Floating button enabled content page is a Tizen custom feature.
    /// So for other platforms need to implement this feature to run on those platforms
    /// </summary>
    public class AlarmListPage : ContentPage
    {
        /// <summary>
        /// Alarm list ui ListView
        /// </summary>
        public AlarmListUI alarmListUI;

        /// <summary>
        /// blank layout to show when no alarm available
        /// </summary>
        internal AlarmEmptyPageLayout blankLayout;

        /// <summary>
        /// blank stack to hold empty one 
        /// </summary>
        internal StackLayout blankStack;

        /// <summary>
        /// blank layout to show when no alarm available
        /// </summary>
        public AlarmListPage()
        {
            /// Title for this page
            Title = "Alarm";
            /// Icon to be shown in the main tab
            Icon = "maintabbed/clock_tabs_ic_alarm.png";

            // Deserialize AlarmRecordDictionary
#if CROSS_PLATFORM
            IAlarmPersistentHandler serizliazer = DependencyService.Get<IAlarmPersistentHandler>();
            // Need to retrieve at the page creation time
            AlarmRecord.AlarmRecordDictionary = serizliazer.DeserializeAlarmRecord();
#else
            // Need to retrieve at the page creation time
            AlarmRecord.AlarmRecordDictionary = Clock.Tizen.Impls.AlarmPersistentHandler.GetInstance().DeserializeAlarmRecord();
#endif
            // Create alarmList UI whether or not there is a saved alarm record
            alarmListUI = new AlarmListUI();
            // Decide what to show blank or list
            if (AlarmRecord.AlarmRecordDictionary == null || AlarmRecord.AlarmRecordDictionary.Count == 0)
            {
                /// if blank stack is null, create a new one
                if (blankStack == null)
                {
                    /// Constructs blank stack UI
                    CreateBlankStack();
                }

                /// Assigns to this page content
                Content = blankStack;
            }
            else
            {
                /// If there is already alarm record, it means alarmListUI has been created. So just show it
                Content = alarmListUI;
            }
        }

        /// <summary>
        /// Called when the floating button has been clicked
        /// Show a page for editing an alarm
        /// When the floating button is clicked (new alarm request), new alarm will be created and call
        /// Alarm Edit Page with the created alarm record
        /// </summary>
        /// <param name="sender">floating button object</param>
        /// <seealso cref="System.object">
        /// <param name="e">Event argument for event of floating button.</param>
        /// <seealso cref="System.EventArgs">
        public void OnFloatingButtonClicked(object sender, EventArgs e)
        {
            /// Creates default alarm record
            AlarmRecord defaultAlarmRecord = new AlarmRecord();
            System.Diagnostics.Debug.WriteLine("                               AlarmListPage.OnFloatingButtonClicked-->EditPage");
            /// Call via alarm page controller which instantiates page once
            Navigation.PushAsync(AlarmPageController.GetInstance(AlarmPages.EditPage, defaultAlarmRecord), false);
        }

        /// <summary>
        /// Each time this page is appearing, alarm count should be checked and if no alarm, 
        /// then use the blank page instead of alarm list view.
        /// In other case, just show alarm list UI.
        /// </summary>
        protected override void OnAppearing()
        {
            // When this page is shown, a floating button should be visible for an app user to add a new alarm
            ((App)Application.Current).ShowFloatingButton(Title);
            /// Needs to check alarm record dictionary availability
            /// There is also a case no alarm in the list.
            /// In both cases, blank page should be shown.
            if (AlarmRecord.AlarmRecordDictionary == null || AlarmRecord.AlarmRecordDictionary.Count == 0)
            {
                /// Checks availability of the blank stack
                if (blankStack == null)
                {
                    CreateBlankStack();
                }

                /// Assigns to this page
                Content = blankStack;
            }
            else
            {
                /// Assigns to this page
                Content = alarmListUI;
            }
        }

        /// <summary>
        /// Each time this page is disappearing, a floating button should be hidden.
        /// </summary>
        protected override void OnDisappearing()
        {
            ((App)Application.Current).HideFloatingButton(Title);
        }

        /// <summary>
        /// Blank stack looks trivial but still needs to align text in the middle of the page
        /// So you can't just put a big label to show this message since labels should be
        /// located into proper location
        /// </summary>
        private void CreateBlankStack()
        {
            /// Create a blan page layout
            blankLayout = new AlarmEmptyPageLayout();
            /// Sets proper texts
            blankLayout.MainTitle = "No alarms";
            blankLayout.Subline1 = "After you create alarms, they will";
            blankLayout.Subline2 = "be shown here";
            /// layout options to be center since blank layout should be placed in the stack
            blankLayout.VerticalOptions = LayoutOptions.Center;

            /// Creates stack layout to hold blank layout
            blankStack = new StackLayout()
            {
                Children =
                {
                    blankLayout
                },
                /// Needs to fill this page vertically
                VerticalOptions = LayoutOptions.FillAndExpand,
                /// Needs to fill this page horizontally
                HorizontalOptions = LayoutOptions.FillAndExpand,
                /// Sets background to white
                BackgroundColor = Color.White
            };
        }
    }
}
