/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using System;

using Xamarin.Forms;

namespace CalculatorUI
{
    /// <summary>
    /// Main page of CalculatorUI app
    /// It is the UI part.
    /// This app layout UI mainly by using Grid.
    /// </summary>
    public partial class CalculatorPage
    {
        /// <summary>
        /// Create UI components.
        /// </summary>
        void InitializeComponent()
        {

            //create main container
            _mainStack = new StackLayout()
            {
                HorizontalOptions = LayoutOptions.FillAndExpand,
                VerticalOptions = LayoutOptions.FillAndExpand,
            };

            //create panel which contain Cal's keyPad
            _panel = CreatePanel();

            //add _sreen in the top of _mainGrid
            _mainStack.Children.Add(App._screen);

            //add _panel at the bottom of _mainGrid
            _mainStack.Children.Add(_panel);

            Content = _mainStack;
        }

        /// <summary>
        /// Click Call back of buttons
        /// </summary>
        /// <param name="sender">sender object</param>
        /// <param name="e">event</param>
        private void CharClicked(object sender, EventArgs e)
        {
            Button en = sender as Button;
            //append new Character
            if (App._screen.Text.Length < 20)
            {
                App._screen.Text += en.Text;
            }
        }

        /// <summary>
        /// Create basic calculator pad
        /// </summary>
        /// <returns>return Grid layout</returns>
        private Grid CreateBasicPad()
        {
            //Grid layout
            Grid basisPadGrid = new Grid()
            {
            };

            //Add buttons to Grid
            Button btn_C = CreateButton("C");
            btn_C.Clicked += (sender, e) =>
            {
                //"C" button, clear all text.
                App._screen.Text = "";
            };
            basisPadGrid.Children.Add(btn_C, 0, 1, 0, 1);

            Button btn_div = CreateButton("/");
            btn_div.Clicked += CharClicked;
            basisPadGrid.Children.Add(btn_div, 1, 2, 0, 1);

            Button btn_X = CreateButton("X");
            btn_X.Clicked += CharClicked;
            basisPadGrid.Children.Add(btn_X, 2, 3, 0, 1);

            Button btn_back = CreateButton("Back");
            btn_back.Clicked += (sender, e) =>
            {
                //"Back" button, remove last char
                string text = App._screen.Text;
                if (text.Length > 0)
                {
                    App._screen.Text = text.Substring(0, text.Length - 1);
                }
            };
            basisPadGrid.Children.Add(btn_back, 3, 4, 0, 1);

            Button btn_7 = CreateButton("7");
            btn_7.Clicked += CharClicked;
            basisPadGrid.Children.Add(btn_7, 0, 1, 1, 2);

            Button btn_8 = CreateButton("8");
            btn_8.Clicked += CharClicked;
            basisPadGrid.Children.Add(btn_8, 1, 2, 1, 2);

            Button btn_9 = CreateButton("9");
            btn_9.Clicked += CharClicked;
            basisPadGrid.Children.Add(btn_9, 2, 3, 1, 2);

            Button btn_sub = CreateButton("-");
            btn_sub.Clicked += CharClicked;
            basisPadGrid.Children.Add(btn_sub, 3, 4, 1, 2);

            Button btn_4 = CreateButton("4");
            btn_4.Clicked += CharClicked;
            basisPadGrid.Children.Add(btn_4, 0, 1, 2, 3);

            Button btn_5 = CreateButton("5");
            btn_5.Clicked += CharClicked;
            basisPadGrid.Children.Add(btn_5, 1, 2, 2, 3);

            Button btn_6 = CreateButton("6");
            btn_6.Clicked += CharClicked;
            basisPadGrid.Children.Add(btn_6, 2, 3, 2, 3);

            Button btn_add = CreateButton("+");
            btn_add.Clicked += CharClicked;
            basisPadGrid.Children.Add(btn_add, 3, 4, 2, 3);

            Button btn_1 = CreateButton("1");
            btn_1.Clicked += CharClicked;
            basisPadGrid.Children.Add(btn_1, 0, 1, 3, 4);

            Button btn_2 = CreateButton("2");
            btn_2.Clicked += CharClicked;
            basisPadGrid.Children.Add(btn_2, 1, 2, 3, 4);

            Button btn_3 = CreateButton("3");
            btn_3.Clicked += CharClicked;
            basisPadGrid.Children.Add(btn_3, 2, 3, 3, 4);

            Button btn_bracket = CreateButton("( )");
            btn_bracket.Clicked += CharClicked;
            basisPadGrid.Children.Add(btn_bracket, 3, 4, 3, 4);

            Button btn_0 = CreateButton("0");
            btn_0.Clicked += CharClicked;
            basisPadGrid.Children.Add(btn_0, 0, 1, 4, 5);

            Button btn_dot = CreateButton(".");
            btn_dot.Clicked += CharClicked;
            basisPadGrid.Children.Add(btn_dot, 1, 2, 4, 5);

            Button btn_rev = CreateButton("+/-");
            btn_rev.Clicked += CharClicked;
            basisPadGrid.Children.Add(btn_rev, 2, 3, 4, 5);

            Button btn_equ = CreateButton("=");
            btn_equ.Clicked += CharClicked;
            basisPadGrid.Children.Add(btn_equ, 3, 4, 4, 5);

            return basisPadGrid;
        }

        /// <summary>
        /// Create advance calculator pad
        /// </summary>
        /// <returns>return Grid layout</returns>
        private Grid CreateAdvancePad()
        {
            //Grid layout
            Grid adPadGrid = new Grid()
            {
            };

            //add buttons to Grid
            Button btn = CreateButton("x!");
            btn.Clicked += CharClicked;
            adPadGrid.Children.Add(btn, 0, 1, 0, 1);

            Button btn_sqrt = CreateButton("sqrt");
            btn.Clicked += CharClicked;
            adPadGrid.Children.Add(btn_sqrt, 1, 2, 0, 1);

            Button btn_rem = CreateButton("%");
            btn.Clicked += CharClicked;
            adPadGrid.Children.Add(btn_rem, 2, 3, 0, 1);

            Button btn_sin = CreateButton("sin");
            btn_sin.Clicked += CharClicked;
            adPadGrid.Children.Add(btn_sin, 0, 1, 1, 2);

            Button btn_cos = CreateButton("cos");
            btn.Clicked += CharClicked;
            adPadGrid.Children.Add(btn_cos, 1, 2, 1, 2);

            Button btn_tan = CreateButton("tan");
            btn.Clicked += CharClicked;
            adPadGrid.Children.Add(btn_tan, 2, 3, 1, 2);

            Button btn_ln = CreateButton("ln");
            btn.Clicked += CharClicked;
            adPadGrid.Children.Add(btn_ln, 0, 1, 2, 3);

            Button btn_log = CreateButton("log");
            btn.Clicked += CharClicked;
            adPadGrid.Children.Add(btn_log, 1, 2, 2, 3);

            btn = CreateButton("1/x");
            btn.Clicked += CharClicked;
            adPadGrid.Children.Add(btn, 2, 3, 2, 3);

            btn = CreateButton("e^x");
            btn.Clicked += CharClicked;
            adPadGrid.Children.Add(btn, 0, 1, 3, 4);

            btn = CreateButton("X^2");
            btn.Clicked += CharClicked;
            adPadGrid.Children.Add(btn, 1, 2, 3, 4);

            btn = CreateButton("Y^x");
            btn.Clicked += CharClicked;
            adPadGrid.Children.Add(btn, 2, 3, 3, 4);

            Button btn_abs = CreateButton("|x|");
            btn.Clicked += CharClicked;
            adPadGrid.Children.Add(btn_abs, 0, 1, 4, 5);

            Button btn_PI = CreateButton("PI");
            btn.Clicked += CharClicked;
            adPadGrid.Children.Add(btn_PI, 1, 2, 4, 5);

            Button btn_e = CreateButton("e");
            btn.Clicked += CharClicked;
            adPadGrid.Children.Add(btn_e, 2, 3, 4, 5);

            return adPadGrid;
        }

        /// <summary>
        /// Calculator panel, contain calculator pad.
        /// </summary>
        /// <returns>Return Grid layout</returns>
        private View CreatePanel()
        {
            //Create Grid layout
            Grid grid = new Grid()
            {
                HorizontalOptions = LayoutOptions.FillAndExpand,
                VerticalOptions = LayoutOptions.FillAndExpand,
            };

            _basicPad = CreateBasicPad();
            if (_isLandScape == false)
            {
                //Portrait, only display basicPad
                grid.Children.Add(_basicPad, 0, 1, 0, 1);
            }
            else
            {
                //Landscape, layout advance pad on left, basic pad on right.
                _advancePad = CreateAdvancePad();
                grid.Children.Add(_advancePad, 0, 1, 0, 1);
                grid.Children.Add(_basicPad, 1, 2, 0, 1);
            }

            _panelGrid = grid;

            //use stacklayout instead of Panel now
            StackLayout panel = new StackLayout()
            {
                IsVisible = true,
                HorizontalOptions = LayoutOptions.FillAndExpand,
                VerticalOptions = LayoutOptions.FillAndExpand,
                Orientation = StackOrientation.Vertical,
                Padding = new Thickness(10),
                Children = { grid }
            };

            return panel;
        }

        /// <summary>
        /// Create a button on calculator page
        /// </summary>
        /// <param name="text">Text</param>
        /// <returns>Button btn</returns>
        public static Button CreateButton(string text)
        {
            Button btn = new Button();
            btn.Text = text;

            return btn;

        }
    }
}
