﻿/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using System.Collections.Generic;
using Xamarin.Forms;
using ApplicationLayoutSamples.Custom;

namespace ApplicationLayoutSamples.Pages
{
    /// <summary>
    /// A class for ScrollInScroll page.
    /// </summary>
    public class ScrollInScrollPage : ContentPage
    {
        private List<CustomFavorite> favoriteList = new List<CustomFavorite>();

        /// <summary>
        /// Initializes a new instance of the <see cref="ScrollInScrollPage"/> class.
        /// </summary>
        /// <param name="title">title</param>
        public ScrollInScrollPage(string title)
        {
            // Title for this page.
            Label tt = new Label
            {
                Text = title,
                TextColor = Color.White,
                FontSize = 28,
                HeightRequest = App.ScreenHeight * 0.075,
                HorizontalOptions = LayoutOptions.Center
            };

            // 1.Image gallery with scroll view.
            ScrollView imageSv = new ScrollView
            {
                Orientation = ScrollOrientation.Horizontal,
                Content = new StackLayout
                {
                    Orientation = StackOrientation.Horizontal,
                    Children =
                    {
                        CreateImage("clock1.png"),
                        CreateImage("clock2.png"),
                        CreateImage("clock3.png"),
                        CreateImage("clock4.png"),
                        CreateImage("clock5.png"),
                        CreateImage("clock6.png"),
                    }
                }
            };

            // 2.Score layout with 5 custom favorites.
            favoriteList.Add(CreateFavorite("0"));
            favoriteList.Add(CreateFavorite("1"));
            favoriteList.Add(CreateFavorite("2"));
            favoriteList.Add(CreateFavorite("3"));
            favoriteList.Add(CreateFavorite("4"));
            StackLayout scoreLayout = new StackLayout
            {
                Orientation = StackOrientation.Horizontal,
                HorizontalOptions = LayoutOptions.Center,
                Children =
                {
                    favoriteList[0],
                    favoriteList[1],
                    favoriteList[2],
                    favoriteList[3],
                    favoriteList[4]
                }
            };

            // 3.Long label.
            Label longLabel = new Label
            {
                Margin = 20,
                FontSize = 15,
                Text = "Xamarin.Forms is a cross-platform, natively-backed UI toolkit abstraction that allows developers to create user interfaces that can be shared across Android, iOS, and Windows Phone. Because the user interfaces use the native controls of the target platforms, they have the appearance and, importantly, the responsiveness of natively built apps."
                     + "\nXamarin.Forms user interfaces are rendered using the native controls of the target platform, allowing Xamarin.Forms applications to retain the appropriate look and feel for each platform. Custom Renderers let developers override this process to customize the appearance and behavior of Xamarin.Forms controls on each platform."
                     + "\nCustom renderers provide a powerful approach for customizing the appearance and behavior of Xamarin.Forms controls. They can be used for small styling changes or sophisticated platform-specific layout and behavior customization. This article provides an introduction to custom renderers, and outlines the process for creating a custom renderer."
            };

            // Put 3 children into main scroll view.
            ScrollView mainSv = new ScrollView
            {
                Content = new StackLayout
                {
                    BackgroundColor = Color.White,
                    Children =
                    {
                        imageSv,
                        scoreLayout,
                        longLabel
                    }
                }
            };

            // Put title and main scroll view into stack layout.
            Content = new StackLayout()
            {
                Children =
                {
                    tt,
                    mainSv
                }
            };
        }

        /// <summary>
        /// A method to create a new image.
        /// </summary>
        /// <param name="imagePath">The image path</param>
        /// <returns>Return the new image</returns>
        public Image CreateImage(string imagePath)
        {
            Image image = new Image
            {
                Margin = 20,
                WidthRequest = 285,
                HeightRequest = 510,
                Source = imagePath
            };

            return image;
        }

        /// <summary>
        /// A method to create a new favorite.
        /// </summary>
        /// <param name="data">The favorite data</param>
        /// <returns>Return the new favorite</returns>
        public CustomFavorite CreateFavorite(string data)
        {
            CustomFavorite favorite = new CustomFavorite
            {
                Margin = 20,
                AutomationId = data
            };

            favorite.Toggled += (sender, evt) =>
            {
                int i = int.Parse(((CustomFavorite)sender).AutomationId);
                if (!favorite.IsToggled)
                {
                    while (++i < 5)
                    {
                        favoriteList[i].IsToggled = false;
                    };
                }
                else
                {
                    while (--i >= 0)
                    {
                        favoriteList[i].IsToggled = true;
                    }
                }
            };

            return favorite;
        }
    }
}
