﻿/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using Xamarin.Forms;

namespace ApplicationLayoutSamples.Pages
{
    /// <summary>
    /// A class for Editfield page.
    /// </summary>
    public class EditfieldPage : ContentPage
    {
        /// <summary>
        /// Initializes a new instance of the <see cref="EditfieldPage"/> class.
        /// </summary>
        /// <param name="title">title</param>
        public EditfieldPage(string title)
        {
            // Title for this page.
            Label tt = new Label
            {
                Text = title,
                TextColor = Color.White,
                FontSize = 28,
                HeightRequest = App.ScreenHeight * 0.075,
                HorizontalOptions = LayoutOptions.Center
            };
            Content = new StackLayout()
            {
                Children =
                {
                    tt,
                    new StackLayout()
                    {
                        HorizontalOptions = LayoutOptions.FillAndExpand,
                        VerticalOptions = LayoutOptions.FillAndExpand,
                        BackgroundColor = Color.White,
                        Children =
                        {
                            CreateEditfield(EditType.MultiplyLine, "Multiply Line"),
                            CreateUnderline(),
                            CreateEditfield(EditType.SingleLine, "Single Line"),
                            CreateUnderline(),
                            CreateEditfield(EditType.Password, "Password"),
                            CreateUnderline()
                        }
                    }
                }
            };
        }

        /// <summary>
        /// A method for create a new input view like Editor, Entry or Password.
        /// </summary>
        /// <param name="type">The edit type.</param>
        /// <param name="placeholder">The place text.</param>
        /// <returns>Return the new input view.</returns>
        public InputView CreateEditfield(EditType type, string placeholder)
        {
            InputView input;

            // New a Editor when EditType is MultiplyLine.
            if (type == EditType.MultiplyLine)
            {
                input = new Editor
                {
                    Text = placeholder,
                    TextColor = Color.Gray,
                    FontSize = 25
                };
            }
            // New a Entry when EditType is SingleLine
            else if (type == EditType.SingleLine)
            {
                input = new Entry
                {
                    Placeholder = placeholder,
                    FontSize = 25
                };
            }
            // New a Password Entry when EditType is Password
            else
            {
                input = new Entry
                {
                    Placeholder = placeholder,
                    FontSize = 25,
                    IsPassword = true
                };
            }
            input.MinimumHeightRequest = 200;
            input.Margin = new Thickness(20, 0, 20, 0);

            // Binding a method to change Text or Placeholder when trigger Focused event.
            input.Focused += (s, e) =>
            {
                if (type == EditType.MultiplyLine)
                {
                    if (((Editor)input).Text == placeholder)
                    {
                        ((Editor)input).Text = string.Empty;
                        ((Editor)input).TextColor = Color.Black;
                    }
                }
                else
                {
                    if (((Entry)input).Placeholder == placeholder)
                    {
                        ((Entry)input).Placeholder = string.Empty;
                    }
                }
            };

            // Binding a method to change Text or Placeholder when trigger Unfocused event.
            input.Unfocused += (s, e) =>
            {
                if (type == EditType.MultiplyLine)
                {
                    if (((Editor)input).Text == string.Empty)
                    {
                        ((Editor)input).Text = placeholder;
                        ((Editor)input).TextColor = Color.Gray;
                    }
                }
                else
                {
                    if (((Entry)input).Placeholder == string.Empty && ((Entry)input).Text == string.Empty)
                    {
                        ((Entry)input).Placeholder = placeholder;
                    }
                }
            };

            return input;
        }

        /// <summary>
        /// A method for create a new underline.
        /// </summary>
        /// <returns>Return the new underline.</returns>
        public BoxView CreateUnderline()
        {
            BoxView underline = new BoxView
            {
                HeightRequest = 1.5d,
                BackgroundColor = Color.Blue,
                Margin = new Thickness(20, 10, 20, 20)
            };
            return underline;
        }

        /// <summary>
        /// A enumerate for edit type.
        /// </summary>
        public enum EditType
        {
            MultiplyLine = 0,
            SingleLine,
            Password
        }
    }
}
