/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using System;
using System.Collections.Generic;
using Xamarin.Forms;

namespace ApplicationControl
{
    public class App : Application
    {
        internal Label selectedAppLabel;
        internal Button searchButton;
        internal Button killButton;
        internal Button executeButton;
        private bool _contentLoaded;
        IScreenSize ScreenSize;
        IAppOperations AppControlOperations;

        static internal class AppControlChoiceConstants
        {
            internal const string Pick = "Pick Operation";
            internal const string View = "View Operation";
            internal const string Compose = "Compose Operation";
        }

        public App()
        {
            ScreenSize = DependencyService.Get<IScreenSize>();
            AppControlOperations = DependencyService.Get<IAppOperations>();
            // The root page of your application
            MainPage = new ContentPage();

            InitializeComponent();
        }

        /// <summary>
        /// app operation controller
        /// </summary>
        internal class MainPageController
        {
            internal List<Button> OperationList { get; set; }
            internal List<Button> AppList { get; set; }
            internal string SelectedOperation { get; set; }
            internal string SelectedApp { get; set; }
            internal bool HasSomethingToKill { get; set; }
            static internal MainPageController SingleInstance { get; set; }

            /// <summary>
            /// get app operation controller instance
            /// </summary>
            /// <returns>app operation controller instance </returns>
            static internal MainPageController GetInstance()
            {
                if (SingleInstance == null)
                {
                    SingleInstance = new MainPageController();
                }

                return SingleInstance;
            }

            /// <summary>
            /// operation controller construct
            /// </summary>
            private MainPageController()
            {
                Initialization();
            }

            /// <summary>
            /// initialize variables
            /// </summary>
            internal void Initialization()
            {
                SelectedOperation = string.Empty;
                SelectedApp = string.Empty;
                if (OperationList != null)
                {
                    OperationList.Clear();
                }

                if (AppList != null)
                {
                    AppList.Clear();
                }
            }
        }

        /// <summary>
        /// initialize application control ui layout
        /// </summary>
        void InitializeComponent()
        {
            if (_contentLoaded)
            {
                return;
            }

            _contentLoaded = true;

            Label header = new Label()
            {
                Text = "Application Control",
                TextColor = Color.White,
                FontSize = 28,
                HeightRequest = ScreenSize.GetHeight() * 0.075,
                HorizontalOptions = LayoutOptions.Center
            };

            // three kinds of app operation:Pick,View,Compose
            StackLayout operationSelectStack = new StackLayout()
            {
                Spacing = 5,
                Padding = new Thickness(10),
                Children =
                {
                    new Button()
                    {
                        Text = AppControlChoiceConstants.Pick,
                    },
                    new Button()
                    {
                        Text = AppControlChoiceConstants.View,
                    },
                    new Button()
                    {
                        Text = AppControlChoiceConstants.Compose,
                    },
                }
            };
            var radios = operationSelectStack.Children;
            foreach (var radio in radios)
            {
                ((Button)radio).Clicked += (sender, ob) =>
                {
                    Button rb = (Button)sender;
                    if (rb.Text == AppControlChoiceConstants.Pick)
                    {
                        // choose pick operation
                        MainPageController.GetInstance().SelectedOperation = AppControlOperations.Pick();
                    }
                    else if (rb.Text == AppControlChoiceConstants.View)
                    {
                        // choose view operation
                        MainPageController.GetInstance().SelectedOperation = AppControlOperations.View();
                    }
                    else
                    {
                        // choose compose operation
                        MainPageController.GetInstance().SelectedOperation = AppControlOperations.Compose();
                    }
                };
            }
            // show selected app operation ,default "No app selected"
            Label noAppSelectedLabel = new Label()
            {
                Text = "No app selected",
                HeightRequest = GetPixelFromHeightPercent(3),
                WidthRequest = ScreenSize.GetWidth(),
            };

            selectedAppLabel = noAppSelectedLabel;

            searchButton = new Button()
            {
                Text = "Search",
            };
            // search available app operation
            searchButton.Clicked += (sSearch, oSearch) =>
            {
                if (MainPageController.GetInstance().SelectedOperation != string.Empty)
                {
                    IEnumerable<string> applicationIds = AppControlOperations.GetMatchedApplicationIds(
                                                            MainPageController.GetInstance().SelectedOperation);
                    ConstructAppSelectDialog(applicationIds);
                }
            };
            // select app operation
            executeButton = new Button()
            {
                Text = "Execute",
            };
            // execute the selected app operation
            executeButton.Clicked += (sSearch, oSearch) =>
            {
                if (MainPageController.GetInstance().SelectedApp != string.Empty)
                {
                    AppControlOperations.SendLaunchRequest(
                                        MainPageController.GetInstance().SelectedOperation,
                                        MainPageController.GetInstance().SelectedApp);
                    MainPageController.GetInstance().HasSomethingToKill = true;
                    killButton.IsEnabled = true;
                }
            };
            // kill the selected running app operation
            killButton = new Button()
            {
                Text = "Kill",
                IsEnabled = false
            };

            // execute the killButton operation
            killButton.Clicked += (s, e) =>
            {
                noAppSelectedLabel.Text = "No app selected";
                MainPageController.GetInstance().SelectedApp = string.Empty;
                killButton.IsEnabled = false;
            };

            Grid centerBtnGrid = new Grid()
            {
                ColumnSpacing = 3,
                RowDefinitions =
                {
                    new RowDefinition { Height = GetPixelFromHeightPercent(7)},
                    new RowDefinition { Height = GetPixelFromHeightPercent(7)},
                    new RowDefinition { Height = GetPixelFromHeightPercent(7)},
                }
            };

            centerBtnGrid.Children.Add(searchButton, 0, 0);
            centerBtnGrid.Children.Add(executeButton, 1, 0);
            centerBtnGrid.Children.Add(killButton, 2, 0);

            // compose a message label
            Label composeMessage = new Label()
            {
                Text = "Compose a message",
                FontSize = GetPixelFromHeightPercent(2),
                FontAttributes = FontAttributes.Bold,
                HorizontalTextAlignment = TextAlignment.Center,
            };

            StackLayout composeStack = new StackLayout()
            {
                Padding = new Thickness(10, 0, 10, 0),
                Spacing = 10,
                WidthRequest = ScreenSize.GetWidth(),
                Orientation = StackOrientation.Horizontal,
                Children =
                {
                    new Label()
                    {
                        Text = "Address To:",
                        TextColor = Color.Black,
                    },
                    //email address
                    new Entry()
                    {
                        BackgroundColor = Color.Gray,
                        Text = "example@tizen.org",
                        VerticalOptions = LayoutOptions.FillAndExpand,
                        HorizontalTextAlignment = TextAlignment.Center,
                    }
                },
            };

            // email content
            StackLayout multiEditStack = new StackLayout()
            {
                BackgroundColor = Color.Gray,
                Orientation = StackOrientation.Vertical,
                HeightRequest = GetPixelFromHeightPercent(17),
                Margin = new Thickness(0, 10, 0, 0),
                Padding = new Thickness(10, 0, 10, 0),
                IsEnabled = false,
                Children =
                {
                    new Editor()
                    {
                        Text = "Dear Developer, " + Environment.NewLine +  Environment.NewLine +
                        "this is the default message sent from " + Environment.NewLine +
                        "appcontrol sample application." + Environment.NewLine +
                        "Feel free to modify this text message in an email composer." + Environment.NewLine +
                        Environment.NewLine + "Best Regards.",
                    }
                }
            };

            //email send button
            Button sendButton = new Button()
            {
                Text = "Send Message",
                HorizontalOptions = LayoutOptions.Center,
                WidthRequest = ScreenSize.GetWidth() / 2,
            };

            ((ContentPage)MainPage).Content = new StackLayout
            {
                Children =
                {
                    header,
                    operationSelectStack,
                    selectedAppLabel,
                    centerBtnGrid,
                    composeMessage,
                    composeStack,
                    multiEditStack,
                    sendButton
                }
            };
        }

        /// <summary>
        /// show all available app operation
        /// </summary>
        /// <param name="applicationIds">all available app operation id</param>
        async void ConstructAppSelectDialog(IEnumerable<string> applicationIds)
        {
            StackLayout appIdsRadio = new StackLayout();
            appIdsRadio.HeightRequest = GetPixelFromHeightPercent(15);
            foreach (var v in applicationIds)
            {
                Button rbs = new Button()
                {
                    Text = v,
                    FontSize = 15,
                    HeightRequest = GetPixelFromHeightPercent(5),
                };
                // choose app operation button
                rbs.Clicked += (o, evt) =>
                {
                    MainPageController.GetInstance().SelectedApp = ((Button)o).Text;
                    selectedAppLabel.Text = MainPageController.GetInstance().SelectedApp;
                    MainPage.Navigation.PopModalAsync();
                };
                appIdsRadio.Children.Add(rbs);
            }

            ContentPage NaviPage = new ContentPage()
            {
                Title = "Application List",
                Content = new StackLayout()
                {
                    Orientation = StackOrientation.Vertical,
                    HorizontalOptions = LayoutOptions.FillAndExpand,
                    MinimumHeightRequest = 300,
                    MinimumWidthRequest = 200,
                    Padding = new Thickness(10, 0, 0, 0),
                    Spacing = 10,
                    Children =
                    {
                        appIdsRadio
                    }
                }
            };
            await  MainPage.Navigation.PushModalAsync(NaviPage);
        }

        private int GetPixelFromWidthPercent(int percent)
        {
            return ScreenSize.GetWidth() * percent / 100;
        }

        private int GetPixelFromHeightPercent(int percent)
        {
            return ScreenSize.GetHeight() * percent / 100;
        }

        protected override void OnStart()
        {
            // Handle when your app starts
        }

        protected override void OnSleep()
        {
            // Handle when your app sleeps
        }

        protected override void OnResume()
        {
            // Handle when your app resumes
        }

    }
}
