#include "animation.h"

static void
_menu_button_clicked(void *data, Evas_Object *obj, void *event_info)
{
	evas_object_show(data);
}

static void
_freeze_button_clicked(void *data, Evas_Object *obj, void *event_info)
{
	ecore_animator_freeze(data);
}

static void
evas_object_rotate(Evas_Object *object, double degree){
	Evas_Map *m;
	Evas_Coord x, y, w, h;
	m = evas_map_new(4);
	evas_object_geometry_get(object, &x, &y, &w, &h);
	evas_map_util_points_populate_from_object(m, object);
	evas_map_util_rotate(m, degree, x + (w / 2), y + (h / 2));
	evas_object_map_set(object, m);
	evas_object_map_enable_set(object, EINA_TRUE);
}

static int
calculate_x_position(double frame){
	return 0.25 * screen_width * atan((0.5 - frame) * 1.2 * M_PI ) + 0.45*screen_width;
}

static int
calculate_y_position(double frame){
	return -0.75 * screen_height * tan((0.5 - frame)) + 0.45 * screen_height;
}

static int
calculate_rotation(double frame){
	return MAX_ROTATION_ANGLE * cos((0.5 - frame)*M_PI);
}

static Eina_Bool
_do_animation(void *data, double pos)
{
	double frame = pos;
	int x, y;
	double rotation_angle;
	frame = ecore_animator_pos_map(pos, animation_type, 0.0, 0.0);

	/* calculate position on curve  */
	x = calculate_x_position(frame);
	y = calculate_y_position(frame);
	rotation_angle = calculate_rotation(frame);

	/*
	 * or if the object is moving in a straight line
	 *
	 * x = 0.675 *screen_width;
	 * y = frame * 0.7 * screen_height  + (0.1 * screen_height);
	 * rotation_angle = 0;
	 */

	evas_object_rotate(data, rotation_angle );
	evas_object_move(data, x, y);

	return EINA_TRUE;
}

static void
register_animation(void *appData){
	appdata_s *ad = appData;
	Ecore_Animator *animator = ecore_animator_timeline_add(ANIM_TIME, _do_animation, ad->ship);
	evas_object_smart_callback_add(ad->freezeButton, "clicked", _freeze_button_clicked, animator);
}

static void
_linear_animation(void *data, Evas_Object *obj, void *event_info)
{
	animation_type = ECORE_POS_MAP_LINEAR;
	register_animation(data);
}

static void
_accelerate_animation(void *data, Evas_Object *obj, void *event_info)
{
	animation_type = ECORE_POS_MAP_ACCELERATE;
	register_animation(data);
}

static void
_decelerate_animation(void *data, Evas_Object *obj, void *event_info)
{
	animation_type = ECORE_POS_MAP_DECELERATE;
	register_animation(data);
}

static void
_sinusoidal_animation(void *data, Evas_Object *obj, void *event_info)
{
	animation_type = ECORE_POS_MAP_SINUSOIDAL;
	register_animation(data);
}

static void
restart_ship(Evas_Object *ship){
	char buf[PATH_MAX];

	snprintf(buf, sizeof(buf), "%s/ship.png", app_get_resource_path());
	evas_object_image_file_set(ship, buf, NULL);
	evas_object_resize(ship, SHIP_IMAGE_WIDTH, SHIP_IMAGE_HEIGHT);
	evas_object_move(ship, calculate_x_position(0), calculate_y_position(0) );
}

static void
win_delete_request_cb(void *data , Evas_Object *obj , void *event_info)
{
	elm_exit();
}

static void
win_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = data;
	/* Let window go to hide state. */
	elm_win_lower(ad->win);
}

static Eina_Bool
keydown_cb(void *data , int type , void *event)
{
	appdata_s *ad = data;
	Ecore_Event_Key *ev = event;
	if (!strcmp(ev->keyname, KEY_END)) {
		/* Let window go to hide state. */
		elm_win_lower(ad->win);
		return ECORE_CALLBACK_DONE;
	}

	return ECORE_CALLBACK_PASS_ON;
}

static void
create_base_gui(appdata_s *ad)
{
	char buf[PATH_MAX];
	int button_width;
	const int button_height = 60;
	const int button_pos_x = 20;

	/* Window */
	ad->win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	elm_win_autodel_set(ad->win, EINA_TRUE);

	evas_object_smart_callback_add(ad->win, "delete,request", win_delete_request_cb, NULL);
	eext_object_event_callback_add(ad->win, EEXT_CALLBACK_BACK, win_back_cb, ad);

	/* Get window dimensions */
	elm_win_screen_size_get (ad->win, NULL, NULL, &screen_width, &screen_height);
	button_width = 0.6 * screen_width;

	/* Add background */
	Evas_Object *background = elm_bg_add(ad->win);
	elm_win_resize_object_add(ad->win, background);
	snprintf(buf, sizeof(buf), "%s/background.jpg", app_get_resource_path());
	elm_bg_file_set(background, buf, NULL);

	/* Add ship */
	ad->ship = evas_object_image_filled_add(evas_object_evas_get(ad->win));
	restart_ship(ad->ship);

	/* add freeze animation button */
	ad->freezeButton = elm_button_add(ad->win);
	elm_object_text_set(ad->freezeButton, "<font_size=30>Freeze animation</font_size>");
	evas_object_resize(ad->freezeButton, button_width, button_height);
	evas_object_move(ad->freezeButton, button_pos_x, 120);

	/* add menu */
	ad->menu = elm_menu_add(ad->win);
	elm_menu_move(ad->menu, 75,75);
	elm_menu_item_add(ad->menu, NULL, NULL, "Linear", _linear_animation, ad );
	elm_menu_item_add(ad->menu, NULL, NULL, "Accelerate", _accelerate_animation, ad);
	elm_menu_item_add(ad->menu, NULL, NULL, "Decelearte", _decelerate_animation, ad);
	elm_menu_item_add(ad->menu, NULL, NULL, "Sinusoidal", _sinusoidal_animation, ad);

	/* add select animation button */
	ad->menuButton = elm_button_add(ad->win);
	elm_object_text_set(ad->menuButton, "<font_size=30>Select animation</font_size>");
	evas_object_smart_callback_add(ad->menuButton, "clicked", _menu_button_clicked, ad->menu);
	evas_object_resize(ad->menuButton, button_width, button_height);
	evas_object_move(ad->menuButton, button_pos_x, 15);

	/* Show objects */
	evas_object_show(background);
	evas_object_show(ad->ship);
	evas_object_show(ad->freezeButton);
	evas_object_show(ad->win);
	evas_object_show(ad->menuButton);

	ecore_event_handler_add(ECORE_EVENT_KEY_DOWN, keydown_cb, NULL);
}

static bool
app_create(void *data)
{
	appdata_s *ad = data;
	create_base_gui(ad);
	return true;
}

static void
app_pause(void *data)
{
	/* Take necessary actions when application becomes invisible. */
}

static void
app_resume(void *data)
{
	/* Take necessary actions when application becomes visible. */
}

static void
app_terminate(void *data)
{
	/* Release all resources. */
}

int
main(int argc, char *argv[])
{
	appdata_s ad = {0,};
	app_event_callback_s event_callback;

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = NULL;

	event_callback.low_memory = NULL;
	event_callback.low_battery = NULL;
	event_callback.device_orientation = NULL;
	event_callback.region_format_changed = NULL;

	int ret = app_efl_main(&argc, &argv, &event_callback, &ad);
	if (ret != APP_ERROR_NONE) {
		LOGE("app_efl_main() is failed. err = %d", ret);
	}

	return ret;
}
