/*
* Copyright (c) 2000-2015 Samsung Electronics Co., Ltd All Rights Reserved
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
*/

/*
 * adlib.h
 *
 *  Created on: Oct 22, 2015
 *      Author: anand.r
 */

/**
 * @mainpage
 * InMobi Ads SDK allows you to monetize your Tizen app with the ads provided by InMobi Server.
 * There are two different types of ads i.e., banner ads and full screen Interstitial ads.<BR>
 * Banner Ads can be Small (320 X 50 pixels) or Large (300 X250 pixels).
 * Interstitial ads size 320 X 480 pixels.<BR>
 * The SDK includes a library (adlib.a) and few header files.<BR>
 * <H3>Privileges for Native apps using Tizen InMobi Ads.</H3>
 * App/Game needs to add below privileges in the tizen-manifest.xml mandatorily to use Adlib functionality.<BR>
 * http://tizen.org/privilege/network.get<BR>
 * http://tizen.org/privilege/internet<BR>
 * http://tizen.org/privilege/appmanager.launch<BR>
 * Note: The minimum Tizen SDK version for integrating the Tizen Ads is 2.3.x or higher.
**/
#ifndef _ADLIB_H_
#define _ADLIB_H_

#include <tizen.h>
#include <Elementary.h>

#ifdef __cplusplus
extern "C" {  // only need to export C interface if
// used by C++ source code
#endif

/**
 * @addtogroup AdLib
 * @{
 */

typedef struct _banner_addata banner_ad_h;

typedef enum {
	ADLIB_AD_NO_ERROR,
	ADLIB_AD_INTERNAL_ERROR = 1,
	ADLIB_AD_ADLIB_NOT_INITIALIZED,
	ADLIB_AD_INVALID_AD_HANDLE,
	ADLIB_AD_INVALID_PARAMETERS,
	ADLIB_AD_INVALID_REQUEST,
	ADLIB_AD_NETWORK_ERROR,
	ADLIB_AD_NO_FILL_ERROR,
}adlib_err;

typedef enum {
	ADLIB_AD_EVENT_REQUEST_FAILED,
	ADLIB_AD_EVENT_REQUEST_SUCCESS,
	ADLIB_AD_EVENT_AD_OPENED,
	ADLIB_AD_EVENT_AD_CLOSED,
}adlib_event_type;

typedef enum {
	ADLIB_APP_NONE_STATE,
	ADLIB_APP_PAUSE_STATE,
	ADLIB_APP_RESUME_STATE,
}adlib_state_type;

typedef enum {
	ADLIB_AD_GENDER_FEMALE,
	ADLIB_AD_GENDER_MALE,
}adlib_gender_e;

/**
 * @addtogroup AdView_BannerAd
 * @{
 */
typedef enum {
	BANNER_AD_POSITION_TOP_LEFT = 0,
	BANNER_AD_POSITION_TOP_CENTER,
	BANNER_AD_POSITION_TOP_RIGHT,
	BANNER_AD_POSITION_CENTER,
	BANNER_AD_POSITION_BOTTOM_LEFT,
	BANNER_AD_POSITION_BOTTOM_CENTER,
	BANNER_AD_POSITION_BOTTOM_RIGHT,
}banner_adposition_e;

typedef enum {
	BANNER_AD_SIZE_SMALL = 15,
	BANNER_AD_SIZE_BIG = 10,
}banner_ad_size_e;



/**
 * @addtogroup AdLib
 * @{
 */

/**
 * @brief Called when the banner ad load is succeeded.
 *
 * @param[in] Evas_Object *banner_ad_obj : banner ad object to be used for this ad instance.
 * @param[in] user_data	The user data passed from the callback registration function
 *
 * @return
 * void
 *
 * @see #banner_ad_event_callback_s
 */
typedef void (*on_banner_request_succeeded_event_cb) (Evas_Object *banner_ad_obj, void *user_data);

/**
 * @brief Called when the banner ad load is failed.
 *
 * @param[in] Evas_Object *banner_ad_obj : banner ad object to be used for this ad instance.
 * @param[in] err	The error code which provides failure details
 * @param[in] user_data	The user data passed from the callback registration function
 *
 * @return
 * void
 *
 * @see #banner_ad_event_callback_s
 */
typedef void (*on_banner_request_failed_event_cb) (Evas_Object *banner_ad_obj, adlib_err err, void *user_data);

/**
 * @brief Called when the banner ad is clicked.
 *
 * @param[in] Evas_Object *banner_ad_obj : banner ad object to be used for this ad instance.
 * @param[in] user_data	The user data passed from the callback registration function
 *
 * @return
 * void
 *
 * @see #banner_ad_event_callback_s
 */
typedef void (*on_banner_ad_opened_event_cb) (Evas_Object *banner_ad_obj, void *user_data);


/**
 * @brief Called when the interstitial ad load is succeeded.
 *
 * @param[in] user_data	The user data passed from the callback registration function
 *
 * @return
 * void
 *
 * @see #interstitial_ad_event_callback_s
 */
typedef void (*on_interstitial_request_succeeded_event_cb) ( void *user_data);

/**
 * @brief Called when the interstitial ad load is failed.
 *
 * @param[in] adlib_err err	: The error code which provides failure details
 * @param[in] user_data	The user data passed from the callback registration function
 *
 * @return
 * void
 *
 * @see #interstitial_ad_event_callback_s
 */
typedef void (*on_interstitial_request_failed_event_cb) (adlib_err err, void *user_data);


/**
 * @brief Called when the interstitial ad is clicked.
 *
 * @param[in] void *user_data	:The user data passed from the callback registration function
 *
 * @return
 * void
 *
 * @see #interstitial_ad_event_callback_s
 */
typedef void (*on_interstitial_ad_opened_event_cb) (void *user_data);

/**
 * @brief Called when the interstitial ad is closed.
 *
 * @param[in] void *user_data	The user data passed from the callback registration function
 *
 * @return
 * void
 *
 * @see #interstitial_ad_event_callback_s
 */
typedef void (*on_interstitial_ad_closed_event_cb) (void *user_data);

/**
 * @brief Called when the network connection is regained.
 *
 * @param[in] user_data	The user data passed from the callback registration function
 *
 * @return
 * void
 *
 * @see #network_connection_listener_s
 */
typedef void (*on_network_request_succeeded_event_cb) (void *user_data);

/**
 * @brief Called when the location request is succeeded.
 *
 * @param[in] user_data	The user data passed from the callback registration function
 *
 * @return
 * void
 *
 * @see #location_connection_listener_s
 */
typedef void (*on_location_request_succeeded_event_cb) (void *user_data);

/**
 * @brief Listener for banner ad network regained.
 *
 * @param[in] user_data	The user data passed from the callback registration function
 *
 * @return
 * void
 *
 */
typedef void (*banner_ad_conn_listener_cb)(void *user_data);

typedef struct
{
	on_banner_request_succeeded_event_cb banner_ad_load_request_succeeded; /**< This callback function is called when the banner ad load is succeeded */
	on_banner_request_failed_event_cb banner_ad_load_request_failed; /**< This callback function is called if the banner ad load fails. */
	on_banner_ad_opened_event_cb banner_ad_opened; /**< This callback function is called once banner ad is clicked. */
} banner_ad_event_callback_s;

typedef struct
{
	on_interstitial_request_succeeded_event_cb interstitial_ad_load_request_succeeded; /**< This callback function is called once the interstitial ad load is success. */
	on_interstitial_request_failed_event_cb interstitial_ad_load_request_failed; /**< This callback function is called once interstitial ad load is failed. */
	on_interstitial_ad_opened_event_cb interstitial_ad_opened; /**< This callback function is called once interstitial ad is clicked. */
	on_interstitial_ad_closed_event_cb interstitial_ad_closed; /**< This callback function is called once interstitial ad is closed. */
} interstitial_ad_event_callback_s;

typedef struct
{
	on_network_request_succeeded_event_cb adlib_network_request_succeeded; /**< This callback function is called when the network connection is regained is succeeded*/
	//on_banner_request_failed_event_cb banner_ad_load_request_failed; /**< This callback function is called if the network connection is lost. */
} network_connection_listener_s;

typedef struct
{
	on_location_request_succeeded_event_cb adlib_location_request_succeeded; /**< This callback function is called when the location request is succeeded*/
	//on_banner_request_failed_event_cb banner_ad_load_request_failed; /**< This callback function is called if the location request load fails. */
} location_connection_listener_s;

/**
 * Initializes adlib and its core.
 * @param[in]
 * char *site_id : site id provided for the application by Inmobi during registration.
 *
 * @retval #ADLIB_AD_NO_ERROR Success.
 * @retval #ADLIB_AD_INTERNAL_ERROR Internal error.
 * @retval #ADLIB_AD_INVALID_PARAMETERS Invalid parameter.
 * @return
 * adlib_err : Returns ADLIB_AD_NO_ERROR on success or with the error occurred.
 *
 */
EXPORT_API adlib_err
adlib_init( const char *site_id);

/**
 * Disconnects adlib core and clears networkconnection.
 *
 * @return
 * void
 *
 */
EXPORT_API void
adlib_deinit();

int _registerNetworkConnectionListener(banner_ad_conn_listener_cb cb, void *userdata);

int _registerRecreateConfigListener(banner_ad_conn_listener_cb cb, void *userdata);

/**
 * Set the user's age for targeting purposes.
 * @param[in] int age: The input age given as preference for fetching ads from server.
 */
EXPORT_API void adlib_set_age(int age);

/**
 * Set the user's gender for targeting purposes.
 * @param[in] adlib_gender_e gender: The input gender given as preference for fetching ads from server.
 */
EXPORT_API void adlib_set_gender(adlib_gender_e gender);

/**
 * Set the user's location for targeting purposes.
 * @param[in] char *location: The input location given as preference for fetching ads from server.
 */
EXPORT_API void adlib_set_location(char *location);

/**
 * Set the user's interests for targeting purposes.
 * @param[in] char *interests: The input interests given as preference for fetching ads from server.
 */
EXPORT_API void adlib_set_interests(char *interests);

/**
 * Set the user's postal code for targeting purposes.
 * @param[in] char *postalCode: The input postalCode given as preference for fetching ads from server.
 */
EXPORT_API void adlib_set_postal_code(char *postalCode);

/**
 * Set the user's area code for targeting purposes.
 * @param[in] char *areaCode: The input areaCode given as preference for fetching ads from server.
 */
EXPORT_API void adlib_set_area_code(char *areaCode);

/**
 * Set the user's date of birth for targeting purposes.
 * @param[in] char *dob: The input date of birth given as preference for fetching ads from server.
 */
EXPORT_API void adlib_set_date_of_birth(char *dob);

/**
 * Set the user's income group for targeting purposes.
 * @param[in] char *income: The input income given as preference for fetching ads from server.
 */
EXPORT_API void adlib_set_income(char *income);

/**
 * Set the user's education for targeting purposes.
 * @param[in] char *education: The input education given as preference for fetching ads from server.
 */
EXPORT_API void adlib_set_education(char *education);

/**
 * Set the user's ethnicity for targeting purposes.
 * @param[in] char *ethnicity: The input ethnicity given as preference for fetching ads from server.
 */
EXPORT_API void adlib_set_ethnicity(char *ethnicity);

void __banner_ad_network_request_succeeded_cb(void *user_data);

void __banner_ad_recreate_config_cb(void *user_data);

/**
 * @addtogroup AdView_BannerAd
 * @{
 */

/**
 * This api will add the banner ad on the parent received.
 * @param[out] Evas_Object * : banner ad object to be used for this ad instance.
 * @param[in] Evas_Object *eo : parent object.
 * @param[in] adlib_adsize_e size : size of the ad to be shown.
 *
 * @return
 * adlib_err : Returns ADLIB_AD_NO_ERROR on success or with the error occurred.
 * @retval #ADLIB_AD_NO_ERROR Success.
 * @retval #ADLIB_AD_INVALID_PARAMETERS Invalid parameter.
 * @retval #ADLIB_AD_ADLIB_NOT_INITIALIZED If adlib is not initialised. i.e adlib_init() not called.
 * @note
 * This will only create the object and will not show ad.
 */
EXPORT_API adlib_err
adview_add_banner_ad(Evas_Object **banner_ad_obj, Evas_Object *eo, banner_ad_size_e size);

/**
 * Loads and shows the ad.
 * Returns the success or failure result to the listener callbacks.
 * @param[in] Evas_Object * : banner ad object to be used for this ad instance.
 * @see OnLoadFailed and OnLoadSucceeded callbacks.
 *
 * @return
 * adlib_err : Returns ADLIB_AD_NO_ERROR on success or with the error occurred.
 * @retval #ADLIB_AD_NO_ERROR Success.
 * @retval #ADLIB_AD_INVALID_AD_HANDLE Invalid Handle passed.
 * @post This function invokes on_banner_request_succeeded_event_cb() on load success or on_banner_request_failed_event_cb() on load failure.
 */
EXPORT_API adlib_err
adview_banner_load(Evas_Object *banner_ad_obj);

/**
 * This api will set the auto refresh feature to either enabled or disabled.
 * If set to disabled ads will not refresh automatically and the app needs to call
 * adview_banner_load for loading a new ad.
 * @param[in] Evas_Object * : banner ad object to be used for this ad instance.
 * @param[in] Eina_Bool enable : Boolean to enable or disable auto refresh feature.
 *
 * @retval #ADLIB_AD_NO_ERROR Success.
 * @retval #ADLIB_AD_INVALID_AD_HANDLE Invalid Handle passed.
 * @return
 * adlib_err : Returns ADLIB_AD_NO_ERROR on success or with the error occurred.
 *
 */
EXPORT_API adlib_err
adview_banner_set_enable_auto_refresh(Evas_Object *banner_ad_obj, Eina_Bool enable);

/**
 * This api will set the auto refresh interval for the ad.
 * Minimum refresh interval should be greater than 10 secs.
 * @param[in] Evas_Object * : banner ad object to be used for this ad instance.
 * @param[in] double duration : Refresh interval value should be >= 10.
 * @retval #ADLIB_AD_NO_ERROR Success.
 * @retval #ADLIB_AD_INVALID_AD_HANDLE Invalid Handle passed.
 * @retval #ADLIB_AD_INVALID_PARAMETERS Invalid parameter.
 * @return
 * adlib_err : Returns ADLIB_AD_NO_ERROR on success or with the error occurred.
 */
EXPORT_API adlib_err
adview_banner_set_refresh_interval(Evas_Object *banner_ad_obj, double duration);

/**
 * Sets the position of the ad.
 * Application can set the position of the ad to the predefined positions defined in adlib_adposition_e
 * @param[in] Evas_Object * : banner ad object to be used for this ad instance.
 * @param[in] banner_adposition_e pos: position of the ad
 *
 * @retval #ADLIB_AD_NO_ERROR Success.
 * @retval #ADLIB_AD_INVALID_AD_HANDLE Invalid Handle passed.
 * @retval #ADLIB_AD_INTERNAL_ERROR Internal error.
 * @return
 * adlib_err : Returns ADLIB_AD_NO_ERROR on success or with the error occurred.
 */
EXPORT_API adlib_err
adview_banner_ad_set_position(Evas_Object *banner_ad_obj, banner_adposition_e pos);

/**
 * Adds the banner ad callback handler.
 * @param[in] Evas_Object * : banner ad object to be used for this ad instance.
 * @param[in] callback The set of callback functions to handle ad lifecycle events
 *
 * @retval #ADLIB_AD_NO_ERROR Success.
 * @retval #ADLIB_AD_INVALID_AD_HANDLE Invalid Handle passed.
 * @retval #ADLIB_AD_INVALID_PARAMETERS Invalid parameter.
 * @return
 * adlib_err : Returns ADLIB_AD_NO_ERROR on success or with the error occurred.
 *
 */
EXPORT_API adlib_err
adview_banner_ad_register_callbacks(Evas_Object *banner_ad_obj, banner_ad_event_callback_s *callback, void *user_data);


/**
 * @addtogroup AdView_InterstitialAd
 * @{
 */

/**
 * This api will add the interstitial ad on the parent received.
 * @param[in]
 * Evas_Object *eo : parent object
 *
 * @retval #ADLIB_AD_NO_ERROR Success.
 * @retval #ADLIB_AD_INTERNAL_ERROR Internal error.
 * @retval #ADLIB_AD_INVALID_PARAMETERS Invalid parameter.
 * @retval #ADLIB_AD_ADLIB_NOT_INITIALIZED If adlib is not initialised. i.e adlib_init() not called.
 *
 * @return
 * adlib_err : Returns ADLIB_AD_NO_ERROR on success or with the error occurred.
 *
 * @note
 * This will only create the object and will not show ad.
 */
EXPORT_API adlib_err
adview_add_interstitial_ad(Evas_Object *eo);

/**
 * Loads the interstitial ad.
 * Returns the success or failure result to the listener callbacks.
 * Please see OnLoadFailed and OnLoadSucceeded callbacks.
 *
 * @retval #ADLIB_AD_NO_ERROR Success.
 * @retval #ADLIB_AD_INTERNAL_ERROR Internal error.
 *
 * @return
 * adlib_err : Returns ADLIB_AD_NO_ERROR on success or with the error occurred.
 */
EXPORT_API adlib_err
adview_interstitial_load(void);

/**
 * Shows the interstitial ad.
 *
 * @retval #ADLIB_AD_NO_ERROR Success.
 * @retval #ADLIB_AD_INTERNAL_ERROR Internal error.
 * @return
 * adlib_err : Returns ADLIB_AD_NO_ERROR on success or with the error occurred.
 */
EXPORT_API adlib_err
adview_interstitial_show(void);

/**
 * Adds the interstitial ad callback handler.
 * @param[in] callback The set of callback functions to handle ad lifecycle events
 *
 * @retval #ADLIB_AD_NO_ERROR Success.
 * @retval #ADLIB_AD_INTERNAL_ERROR Internal error.
 * @retval #ADLIB_AD_INVALID_PARAMETERS Invalid parameter.
 * @retval #ADLIB_AD_ADLIB_NOT_INITIALIZED If adlib is not initialised. i.e adlib_init() not called.
 * @return
 * adlib_err : Returns ADLIB_AD_NO_ERROR on success or with the error occurred.
 *
 */
EXPORT_API adlib_err
adview_interstitial_ad_register_callbacks(interstitial_ad_event_callback_s *callback, void *user_data);

#ifdef __cplusplus
}
#endif

#endif // _ADLIB_H_

