/*
* Copyright (c) 2000-2015 Samsung Electronics Co., Ltd All Rights Reserved
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
*/

/*
 * adlib.h
 *
 *  Created on: Oct 22, 2015
 *      Author: anand.r
 */

/**
 * @mainpage
 * InMobi Ads SDK allows you to monetize your Tizen app with the ads provided by InMobi Server.
 * There are two different types of ads i.e., banner ads and full screen Interstitial ads.<BR>
 * Banner Ads can be Small (320 X 50 pixels) or Large (300 X250 pixels).
 * Interstitial ads size 320 X 480 pixels.<BR>
 * The SDK includes a library (adlib.a) and few header files.<BR>
 * <H3>Privileges for Native apps using Tizen InMobi Ads.</H3>
 * App/Game needs to add below privileges in the tizen-manifest.xml mandatorily to use Adlib functionality.<BR>
 * http://tizen.org/privilege/network.get<BR>
 * http://tizen.org/privilege/internet<BR>
 * http://tizen.org/privilege/appmanager.launch<BR>
 * Note: The minimum Tizen SDK version for integrating the Tizen Ads is 2.3.x or higher.
**/
#ifndef _ADLIB_H_
#define _ADLIB_H_

#include <tizen.h>
#include <Elementary.h>

#ifdef __cplusplus
extern "C" {  // only need to export C interface if
// used by C++ source code
#endif

typedef struct _banner_addata banner_ad_h;

typedef enum {
	ADLIB_AD_NO_ERROR,
	ADLIB_AD_INTERNAL_ERROR = 1,
	ADLIB_AD_ADLIB_NOT_INITIALIZED,
	ADLIB_AD_INVALID_AD_HANDLE,
	ADLIB_AD_INVALID_PARAMETERS,
	ADLIB_AD_INVALID_REQUEST,
	ADLIB_AD_NETWORK_ERROR,
	ADLIB_AD_NO_FILL_ERROR,
}adlib_err;

typedef enum {
	ADLIB_AD_EVENT_REQUEST_FAILED,
	ADLIB_AD_EVENT_REQUEST_SUCCESS,
	ADLIB_AD_EVENT_AD_OPENED,
	ADLIB_AD_EVENT_AD_CLOSED,
}adlib_event_type;

typedef enum {
	ADLIB_APP_NONE_STATE,
	ADLIB_APP_PAUSE_STATE,
	ADLIB_APP_RESUME_STATE,
}adlib_state_type;

typedef enum {
	ADLIB_AD_GENDER_FEMALE,
	ADLIB_AD_GENDER_MALE,
}adlib_gender_e;

typedef enum {
	ABOVE_55,
	BELOW_18,
	BETWEEN_18_AND_20,
	BETWEEN_21_AND_24,
	BETWEEN_25_AND_34,
	BETWEEN_35_AND_54,
}adlib_agegroup_e;

typedef enum {
	ABOVE_USD_150K,
	BELOW_USD_5K,
	BETWEEN_USD_100K_AND_150K,
	BETWEEN_USD_10K_AND_15K,
	BETWEEN_USD_15K_AND_20K,
	BETWEEN_USD_20K_AND_25K,
	BETWEEN_USD_25K_AND_50K,
	BETWEEN_USD_50K_AND_75K,
	BETWEEN_USD_5K_AND_10K,
	BETWEEN_USD_75K_AND_100K,
}adlib_incomegroup_e;

typedef enum {
	COLLEGE_OR_GRADUATE,
	HIGH_SCHOOL_OR_LESS,
	POST_GRADUATE_OR_ABOVE,
}adlib_education_e;

typedef enum {
	AFRICAN_AMERICAN,
	ASIAN,
	CAUCASIAN,
	HISPANIC,
	OTHER,
}adlib_ethnicity_e;

typedef enum {
	SINGLE,
	DIVORCED,
	ENGAGED,
	RELATIONSHIP,
}adlib_marital_e;

/**
 * @brief Called when the banner ad load is succeeded.
 *
 * @param[in] Evas_Object *banner_ad_obj : banner ad object to be used for this ad instance.
 * @param[in] user_data	The user data passed from the callback registration function
 *
 * @return
 * void
 *
 * @see #banner_ad_event_callback_s
 */
typedef void (*on_banner_request_succeeded_event_cb) (Evas_Object *banner_ad_obj, void *user_data);

/**
 * @brief Called when the banner ad load is failed.
 *
 * @param[in] Evas_Object *banner_ad_obj : banner ad object to be used for this ad instance.
 * @param[in] err	The error code which provides failure details
 * @param[in] user_data	The user data passed from the callback registration function
 *
 * @return
 * void
 *
 * @see #banner_ad_event_callback_s
 */
typedef void (*on_banner_request_failed_event_cb) (Evas_Object *banner_ad_obj, adlib_err err, void *user_data);

/**
 * @brief Called when the banner ad is clicked.
 *
 * @param[in] Evas_Object *banner_ad_obj : banner ad object to be used for this ad instance.
 * @param[in] user_data	The user data passed from the callback registration function
 *
 * @return
 * void
 *
 * @see #banner_ad_event_callback_s
 */
typedef void (*on_banner_ad_opened_event_cb) (Evas_Object *banner_ad_obj, void *user_data);


/**
 * @brief Called when the interstitial ad load is succeeded.
 *
 * @param[in] user_data	The user data passed from the callback registration function
 *
 * @return
 * void
 *
 * @see #interstitial_ad_event_callback_s
 */
typedef void (*on_interstitial_request_succeeded_event_cb) ( void *user_data);

/**
 * @brief Called when the interstitial ad load is failed.
 *
 * @param[in] adlib_err err	: The error code which provides failure details
 * @param[in] user_data	The user data passed from the callback registration function
 *
 * @return
 * void
 *
 * @see #interstitial_ad_event_callback_s
 */
typedef void (*on_interstitial_request_failed_event_cb) (adlib_err err, void *user_data);


/**
 * @brief Called when the interstitial ad is clicked.
 *
 * @param[in] void *user_data	:The user data passed from the callback registration function
 *
 * @return
 * void
 *
 * @see #interstitial_ad_event_callback_s
 */
typedef void (*on_interstitial_ad_opened_event_cb) (void *user_data);

/**
 * @brief Called when the interstitial ad is closed.
 *
 * @param[in] void *user_data	The user data passed from the callback registration function
 *
 * @return
 * void
 *
 * @see #interstitial_ad_event_callback_s
 */
typedef void (*on_interstitial_ad_closed_event_cb) (void *user_data);

/**
 * @brief Called when the network connection is regained.
 *
 * @param[in] user_data	The user data passed from the callback registration function
 *
 * @return
 * void
 *
 * @see #network_connection_listener_s
 */
typedef void (*on_network_request_succeeded_event_cb) (void *user_data);

/**
 * @brief Called when the location request is succeeded.
 *
 * @param[in] user_data	The user data passed from the callback registration function
 *
 * @return
 * void
 *
 * @see #location_connection_listener_s
 */
typedef void (*on_location_request_succeeded_event_cb) (void *user_data);

/**
 * @brief Listener for banner ad network regained.
 *
 * @param[in] user_data	The user data passed from the callback registration function
 *
 * @return
 * void
 *
 */
typedef void (*banner_ad_conn_listener_cb)(void *user_data);

typedef struct
{
	on_banner_request_succeeded_event_cb banner_ad_load_request_succeeded; /**< This callback function is called when the banner ad load is succeeded */
	on_banner_request_failed_event_cb banner_ad_load_request_failed; /**< This callback function is called if the banner ad load fails. */
	on_banner_ad_opened_event_cb banner_ad_opened; /**< This callback function is called once banner ad is clicked. */
} banner_ad_event_callback_s;

typedef struct
{
	on_interstitial_request_succeeded_event_cb interstitial_ad_load_request_succeeded; /**< This callback function is called once the interstitial ad load is success. */
	on_interstitial_request_failed_event_cb interstitial_ad_load_request_failed; /**< This callback function is called once interstitial ad load is failed. */
	on_interstitial_ad_opened_event_cb interstitial_ad_opened; /**< This callback function is called once interstitial ad is clicked. */
	on_interstitial_ad_closed_event_cb interstitial_ad_closed; /**< This callback function is called once interstitial ad is closed. */
} interstitial_ad_event_callback_s;

typedef struct
{
	on_network_request_succeeded_event_cb adlib_network_request_succeeded; /**< This callback function is called when the network connection is regained is succeeded*/
	//on_banner_request_failed_event_cb banner_ad_load_request_failed; /**< This callback function is called if the network connection is lost. */
} network_connection_listener_s;

typedef struct
{
	on_location_request_succeeded_event_cb adlib_location_request_succeeded; /**< This callback function is called when the location request is succeeded*/
	//on_banner_request_failed_event_cb banner_ad_load_request_failed; /**< This callback function is called if the location request load fails. */
} location_connection_listener_s;

/**
 * Initializes adlib and its core.
 * @param[in]
 * char *site_id : site id provided for the application by Inmobi during registration.
 *
 * @retval #ADLIB_AD_NO_ERROR Success.
 * @retval #ADLIB_AD_INTERNAL_ERROR Internal error.
 * @retval #ADLIB_AD_INVALID_PARAMETERS Invalid parameter.
 * @return
 * adlib_err : Returns ADLIB_AD_NO_ERROR on success or with the error occurred.
 *
 */
EXPORT_API adlib_err
adlib_init( const char *site_id);

/**
 * Disconnects adlib core and clears networkconnection.
 *
 * @return
 * void
 *
 */
EXPORT_API void
adlib_deinit();

int _registerNetworkConnectionListener(banner_ad_conn_listener_cb cb, void *userdata);

int _registerRecreateConfigListener(banner_ad_conn_listener_cb cb, void *userdata);

/**
 * Set the user's gender for targeting purposes.
 * @param[in] adlib_gender_e gender: The input gender given as preference for fetching ads from server.
 */
EXPORT_API void adlib_set_gender(adlib_gender_e gender);

/**
 * Set the user's age group for targeting purposes.
 * @param[in] int year: The input year of birth given as preference for fetching ads from server.
 */
EXPORT_API void adlib_set_year_of_birth(int year);

/**
 * Set the user's income group for targeting purposes.
 * @param[in] int income: The input income value given as preference for fetching ads from server.
 */
EXPORT_API void adlib_set_income(int income);

/**
 * Set the user's education for targeting purposes.
 * @param[in] adlib_education_e education: The input education given as preference for fetching ads from server.
 */
EXPORT_API void adlib_set_education(adlib_education_e education);

/**
 * Set the user's ethnicity for targeting purposes.
 * @param[in] adlib_ethnicity_e ethnicity: The input ethnicity given as preference for fetching ads from server.
 */
EXPORT_API void adlib_set_ethnicity(adlib_ethnicity_e ethnicity);

/**
 * Set the user's marital status for targeting purposes.
 * @param[in] adlib_marital_e marital: The input marital status given as preference for fetching ads from server.
 */
EXPORT_API void adlib_set_marital_status(adlib_marital_e marital);

void __banner_ad_network_request_succeeded_cb(void *user_data);

void __banner_ad_recreate_config_cb(void *user_data);

#ifdef __cplusplus
}
#endif

#endif // _ADLIB_H_

