/*
* Copyright (c) 2000-2015 Samsung Electronics Co., Ltd All Rights Reserved
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
*/

/*
 * adlib_banner_adview.h
 *
 *  Created on: Oct 22, 2015
 *      Author: anand.r
 */

#ifndef ADLIB_BANNER_ADVIEW_H_
#define ADLIB_BANNER_ADVIEW_H_

#include "adlib.h"

#ifdef __cplusplus
extern "C" {  // only need to export C interface if
// used by C++ source code
#endif

typedef enum {
	BANNER_AD_POSITION_TOP_LEFT = 0,
	BANNER_AD_POSITION_TOP_CENTER,
	BANNER_AD_POSITION_TOP_RIGHT,
	BANNER_AD_POSITION_CENTER,
	BANNER_AD_POSITION_BOTTOM_LEFT,
	BANNER_AD_POSITION_BOTTOM_CENTER,
	BANNER_AD_POSITION_BOTTOM_RIGHT,
}banner_adposition_e;

typedef enum {
	BANNER_AD_SIZE_SMALL = 15,
	BANNER_AD_SIZE_BIG = 10,
}banner_ad_size_e;

/**
 * This api will add the banner ad on the parent received.
 * @param[out] Evas_Object * : banner ad object to be used for this ad instance.
 * @param[in] Evas_Object *eo : parent object.
 * @param[in] adlib_adsize_e size : size of the ad to be shown.
 *
 * @return
 * adlib_err : Returns ADLIB_AD_NO_ERROR on success or with the error occurred.
 * @retval #ADLIB_AD_NO_ERROR Success.
 * @retval #ADLIB_AD_INVALID_PARAMETERS Invalid parameter.
 * @retval #ADLIB_AD_ADLIB_NOT_INITIALIZED If adlib is not initialised. i.e adlib_init() not called.
 * @note
 * This will only create the object and will not show ad.
 */
EXPORT_API adlib_err
adview_add_banner_ad(Evas_Object **banner_ad_obj, Evas_Object *eo, banner_ad_size_e size);

/**
 * Loads and shows the ad.
 * Returns the success or failure result to the listener callbacks.
 * @param[in] Evas_Object * : banner ad object to be used for this ad instance.
 * @see OnLoadFailed and OnLoadSucceeded callbacks.
 *
 * @return
 * adlib_err : Returns ADLIB_AD_NO_ERROR on success or with the error occurred.
 * @retval #ADLIB_AD_NO_ERROR Success.
 * @retval #ADLIB_AD_INVALID_AD_HANDLE Invalid Handle passed.
 * @post This function invokes on_banner_request_succeeded_event_cb() on load success or on_banner_request_failed_event_cb() on load failure.
 */
EXPORT_API adlib_err
adview_banner_load(Evas_Object *banner_ad_obj);

/**
 * This api will set the auto refresh feature to either enabled or disabled.
 * If set to disabled ads will not refresh automatically and the app needs to call
 * adview_banner_load for loading a new ad.
 * @param[in] Evas_Object * : banner ad object to be used for this ad instance.
 * @param[in] Eina_Bool enable : Boolean to enable or disable auto refresh feature.
 *
 * @retval #ADLIB_AD_NO_ERROR Success.
 * @retval #ADLIB_AD_INVALID_AD_HANDLE Invalid Handle passed.
 * @return
 * adlib_err : Returns ADLIB_AD_NO_ERROR on success or with the error occurred.
 *
 */
EXPORT_API adlib_err
adview_banner_set_enable_auto_refresh(Evas_Object *banner_ad_obj, Eina_Bool enable);

/**
 * This api will set the auto refresh interval for the ad.
 * Minimum refresh interval should be greater than 10 secs.
 * @param[in] Evas_Object * : banner ad object to be used for this ad instance.
 * @param[in] double duration : Refresh interval value should be >= 10.
 * @retval #ADLIB_AD_NO_ERROR Success.
 * @retval #ADLIB_AD_INVALID_AD_HANDLE Invalid Handle passed.
 * @retval #ADLIB_AD_INVALID_PARAMETERS Invalid parameter.
 * @return
 * adlib_err : Returns ADLIB_AD_NO_ERROR on success or with the error occurred.
 */
EXPORT_API adlib_err
adview_banner_set_refresh_interval(Evas_Object *banner_ad_obj, double duration);

/**
 * Sets the position of the ad.
 * Application can set the position of the ad to the predefined positions defined in adlib_adposition_e
 * @param[in] Evas_Object * : banner ad object to be used for this ad instance.
 * @param[in] banner_adposition_e pos: position of the ad
 *
 * @retval #ADLIB_AD_NO_ERROR Success.
 * @retval #ADLIB_AD_INVALID_AD_HANDLE Invalid Handle passed.
 * @retval #ADLIB_AD_INTERNAL_ERROR Internal error.
 * @return
 * adlib_err : Returns ADLIB_AD_NO_ERROR on success or with the error occurred.
 */
EXPORT_API adlib_err
adview_banner_ad_set_position(Evas_Object *banner_ad_obj, banner_adposition_e pos);

/**
 * Adds the banner ad callback handler.
 * @param[in] Evas_Object * : banner ad object to be used for this ad instance.
 * @param[in] callback The set of callback functions to handle ad lifecycle events
 *
 * @retval #ADLIB_AD_NO_ERROR Success.
 * @retval #ADLIB_AD_INVALID_AD_HANDLE Invalid Handle passed.
 * @retval #ADLIB_AD_INVALID_PARAMETERS Invalid parameter.
 * @return
 * adlib_err : Returns ADLIB_AD_NO_ERROR on success or with the error occurred.
 *
 */
EXPORT_API adlib_err
adview_banner_ad_register_callbacks(Evas_Object *banner_ad_obj, banner_ad_event_callback_s *callback, void *user_data);

#ifdef __cplusplus
}
#endif

#endif /* ADLIB_BANNER_ADVIEW_H_ */
